﻿using System.Collections;
using System.Collections.Generic;
using UnityEngine;
using UnityEngine.UI;
using TMPro;
using UnityEngine.EventSystems;
using System;

[System.Serializable]
public class TextItemsSelector : Selectable, ISelectHandler, IDeselectHandler, IPointerEnterHandler, IPointerExitHandler
{
    //Event in editor to allow code to run when selected
    [SerializeField]
    private Button.ButtonClickedEvent _OnClick = new Button.ButtonClickedEvent();

    public Button.ButtonClickedEvent onClick
    {
        get
        {
            return _OnClick;
        }

        set
        {
            _OnClick = value;
        }
    }

    #region EventArgs Classes
    public class SelectedItemChangedEventArgs : EventArgs
    {
        public string OldItemText { get; set; }
        public string NewItemText { get; set; }
        public int OldIndex { get; set; }
        public int NewIndex { get; set; }
    }

    public event EventHandler<SelectedItemChangedEventArgs> SelectedItemChanged;
    #endregion

    [SerializeField]
    public string[] Items;  //The text items to display
    public int DefaultIndex = 0;    //Allows changing the default displayed text

    //UI controls
    public TMP_Text Text;
    public Button LeftArrow;
    public Button RightArrow;

    //Animations
    private Animator _Animator;
    private Animator _LeftArrowAnimator;
    private Animator _RightArrowAnimator;

    public int CurrentIndex { get; private set; }   //The index of the currently visible text
    public string SelectedItemText { get; private set; }    //The currently visible text
    
    protected override void Awake()
    {
        base.Awake();
        CurrentIndex = DefaultIndex;

        //Get components, subscribe to events
        LeftArrow.onClick.AddListener(OnLeftArrowClick);
        RightArrow.onClick.AddListener(OnRightArrowClick);
        _Animator = GetComponent<Animator>();
        _LeftArrowAnimator = LeftArrow.GetComponent<Animator>();
        _RightArrowAnimator = RightArrow.GetComponent<Animator>();

        //Sanity checks, then display the desired text
        if (CurrentIndex >= 0 && CurrentIndex < Items.Length)
        {
            SelectedItemText = Items[CurrentIndex];
            Text.SetText(SelectedItemText);
        }

        InputManager.Instance.InputStateChanged += OnInputStateChanged;
    }

    // Update is called once per frame
    void Update()
    {
        if (IsInteractable() && currentSelectionState == SelectionState.Selected)
        {
            float val = InputManager.Instance.GetControllerAxisUnique("Controller D-Pad X");

            if (val != 0.0f)
            {
                if (val < -0.2f)
                {
                    //Moved the D-Pad left
                    OnLeftArrowClick();
                }

                else if (val > 0.2f)
                {
                    //Moved the D-Pad right
                    OnRightArrowClick();
                }
            }

            else
            {
                val = InputManager.Instance.GetControllerAxisUnique("Controller Left Stick X");

                if (val != 0.0f)
                {
                    if (val < 0.0f)
                    {
                        //Moved the stick left
                        OnLeftArrowClick();
                    }

                    else if (val > 0.0f)
                    {
                        //Moved the stick right
                        OnRightArrowClick();
                    }
                }

                else if (Input.GetKeyDown(KeyCode.Return) || Input.GetKeyDown("joystick button 0"))
                {
                    //Selected so play the sound and call the event
                    AudioManager.Instance.PlaySFX(AudioManager.Instance.UISelectClip);
                    onClick.Invoke();
                    return;
                }
            }
        }
    }

    private void OnInputStateChanged(object sender, InputManager.InputStateChangedEventArgs e)
    {
        if (e.NewState == Constants.InputState.MouseAndKeyboard || PageManager.CurrentPage.DefaultFocusedControl == null || PageManager.CurrentPage.DefaultFocusedControl.gameObject == null || PageManager.CurrentPage.DefaultFocusedControl.gameObject != gameObject)
        {
            //We're using mouse and keyboard, don't have a page or aren't the focused control so we've lost focus
            OnLostFocus();
        }

        else
        {
            if (e.NewState == Constants.InputState.Controller)
            {
                //We're using a controller and must have focus so call OnFocus
                OnFocus();
            }
        }
    }

    private void OnLeftArrowClick()
    {
        int oldIndex = 0;

        //Play the animation, decrement the indexes, prep the event args
        _LeftArrowAnimator.SetTrigger("Pressed");
        if (CurrentIndex - 1 < 0)
        {
            oldIndex = 0;
            CurrentIndex = Items.Length;
        }

        else
        {
            oldIndex = CurrentIndex;
        }

        CurrentIndex--;

        //Update the UI, play the audio
        SelectedItemText = Items[CurrentIndex];
        Text.SetText(SelectedItemText);
        AudioManager.Instance.PlaySFX(AudioManager.Instance.UIFocusClip);

        //Setup the event args and send the event notification to all subscribers
        SelectedItemChangedEventArgs args = new SelectedItemChangedEventArgs();

        args.OldItemText = Items[oldIndex];
        args.NewItemText = SelectedItemText;
        args.OldIndex = oldIndex;
        args.NewIndex = CurrentIndex;

        OnSelectedItemChanged(args);
    }

    private void OnRightArrowClick()
    {
        int oldIndex = 0;

        //Play the animation, increment the indexes, prep the event args
        _RightArrowAnimator.SetTrigger("Pressed");
        if (CurrentIndex + 1 == Items.Length)
        {
            oldIndex = Items.Length - 1;
            CurrentIndex = -1;
        }

        else
        {
            oldIndex = CurrentIndex;
        }

        CurrentIndex++;

        //Update the UI, play the audio
        SelectedItemText = Items[CurrentIndex];
        Text.SetText(SelectedItemText);
        AudioManager.Instance.PlaySFX(AudioManager.Instance.UIFocusClip);


        //Setup the event args and send the event notification to all subscribers
        SelectedItemChangedEventArgs args = new SelectedItemChangedEventArgs();

        args.OldItemText = Items[oldIndex];
        args.NewItemText = SelectedItemText;
        args.OldIndex = oldIndex;
        args.NewIndex = CurrentIndex;

        OnSelectedItemChanged(args);
    }

    private void OnFocus()
    {
        if (_Animator != null)
        {
            //Reset the triggers
            _Animator.ResetTrigger("OnLostFocus");
            _Animator.SetTrigger("OnFocus");
        }
    }

    private void OnLostFocus()
    {
        if (_Animator != null)
        {
            //Reset the triggers
            _Animator.ResetTrigger("OnFocus");
            _Animator.SetTrigger("OnLostFocus");
        }
    }

    public override void OnPointerEnter(PointerEventData data)
    {
        OnFocus();
    }

    public override void OnPointerExit(PointerEventData data)
    {
        OnLostFocus();
    }

    public override void OnSelect(BaseEventData data)
    {
        OnFocus();
        base.OnSelect(data);
    }

    public override void OnDeselect(BaseEventData data)
    {
        OnLostFocus();
        base.OnDeselect(data);
    }

    protected override void OnDestroy()
    {
        //Unsubscribe from events
        if (InputManager.Instance != null)
        {
            InputManager.Instance.InputStateChanged -= OnInputStateChanged;
        }

        base.OnDestroy();
    }

    private void OnSelectedItemChanged(SelectedItemChangedEventArgs e)
    {
        //Called when the selected item changes and sends an event notification to all subscribers
        EventHandler<SelectedItemChangedEventArgs> handler = SelectedItemChanged;
        if (handler != null)
        {
            handler(this, e);
        }
    }
}
